<?php

namespace App\Http\Controllers;

use App\Models\IndicatorObservation;
use App\Models\GiiVersion;
use App\Models\PillarScore;
use App\Models\NationalScore;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class IndicatorObservationController extends Controller
{
    public function index(Request $request)
    {
        $q = IndicatorObservation::with([
                'indicator:id,code,name',
                'submission:id,status,year,month,institution_id',
            ])
            ->orderByDesc('year')
            ->orderByDesc('month');

        if ($indicatorId = $request->query('gii_indicator_id')) {
            $q->where('gii_indicator_id', $indicatorId);
        }
        if ($submissionId = $request->query('submission_id')) {
            $q->where('submission_id', $submissionId);
        }
        if ($year = $request->query('year')) {
            $q->where('year', (int)$year);
        }
        if ($status = $request->query('status')) {
            $q->where('status', $status);
        }

        $result = $q->paginate($request->query('per_page', 15));

        return response()->json([
            'message' => 'Indicator observations retrieved successfully.',
            'data' => $result,
        ]);
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'gii_indicator_id' => ['required', 'exists:gii_indicators,id'],
            'submission_id' => ['nullable', 'exists:submissions,id'],
            'year' => ['required', 'integer', 'min:1900', 'max:2100'],
            'month' => ['nullable', 'integer', 'min:1', 'max:12'],
            'value' => ['nullable', 'numeric'],
            'unit' => ['nullable', 'string', 'max:50'],
            'status' => ['nullable', 'in:DRAFT,APPROVED,PUBLISHED'],
            'confidence' => ['nullable', 'numeric', 'min:0', 'max:100'],
            'meta' => ['nullable', 'array'],
        ]);

        $observation = IndicatorObservation::create([
            'gii_indicator_id' => $data['gii_indicator_id'],
            'submission_id' => $data['submission_id'] ?? null,
            'year' => $data['year'],
            'month' => $data['month'] ?? null,
            'value' => $data['value'] ?? null,
            'unit' => $data['unit'] ?? null,
            'status' => $data['status'] ?? 'DRAFT',
            'confidence' => $data['confidence'] ?? null,
            'meta' => $data['meta'] ?? null,
        ]);

        return response()->json([
            'message' => 'Indicator observation created successfully.',
            'data' => $observation,
        ], 201);
    }

    public function show(IndicatorObservation $indicatorObservation)
    {
        $indicatorObservation->load(['indicator', 'submission']);

        return response()->json([
            'message' => 'Indicator observation retrieved successfully.',
            'data' => $indicatorObservation,
        ]);
    }

    public function update(Request $request, IndicatorObservation $indicatorObservation)
    {
        $data = $request->validate([
            'gii_indicator_id' => ['sometimes', 'exists:gii_indicators,id'],
            'submission_id' => ['nullable', 'exists:submissions,id'],
            'year' => ['sometimes', 'integer', 'min:1900', 'max:2100'],
            'month' => ['nullable', 'integer', 'min:1', 'max:12'],
            'value' => ['nullable', 'numeric'],
            'unit' => ['nullable', 'string', 'max:50'],
            'status' => ['sometimes', 'in:DRAFT,APPROVED,PUBLISHED'],
            'confidence' => ['nullable', 'numeric', 'min:0', 'max:100'],
            'meta' => ['nullable', 'array'],
        ]);

        $indicatorObservation->update($data);

        return response()->json([
            'message' => 'Indicator observation updated successfully.',
            'data' => $indicatorObservation->fresh(),
        ]);
    }

    public function destroy(IndicatorObservation $indicatorObservation)
    {
        $indicatorObservation->delete();

        return response()->json([
            'message' => 'Indicator observation deleted successfully.',
        ]);
    }

     public function compute($year)
    {
        $version = GiiVersion::where('year', $year)->firstOrFail();

        DB::transaction(function () use ($version, $year) {
            foreach ($version->pillars()->with('subpillars.indicators')->get() as $pillar) {
                $indicatorIds = $pillar->subpillars->flatMap(fn($s) => $s->indicators->pluck('id'))->values();

                $avg = DB::table('indicator_observations')
                    ->whereIn('gii_indicator_id', $indicatorIds)
                    ->where('year', (int)$year)
                    ->where('status', 'APPROVED')
                    ->avg('value');

                PillarScore::updateOrCreate(
                    ['gii_pillar_id' => $pillar->id, 'year' => (int)$year],
                    ['score' => $avg ?? 0, 'meta' => ['method' => 'avg-approved-observations']]
                );
            }

            $pillarAvg = DB::table('pillar_scores')
                ->whereIn('gii_pillar_id', $version->pillars()->pluck('id'))
                ->where('year', (int)$year)
                ->avg('score');

            NationalScore::updateOrCreate(
                ['gii_version_id' => $version->id, 'year' => (int)$year],
                ['score' => $pillarAvg ?? 0, 'meta' => ['method' => 'avg-pillar-scores']]
            );
        });

        return response()->json([
            'message' => "Analytics computed for {$year}.",
        ]);
    }

    public function pillarScores($year)
    {
        $version = GiiVersion::where('year', $year)->firstOrFail();

        $scores = PillarScore::with('pillar')
            ->whereIn('gii_pillar_id', $version->pillars()->pluck('id'))
            ->where('year', (int)$year)
            ->get();

        return response()->json([
            'message' => 'Pillar scores retrieved successfully.',
            'data' => $scores
        ]);
    }

    public function nationalScore($year)
    {
        $version = GiiVersion::where('year', $year)->firstOrFail();

        $score = NationalScore::where('gii_version_id', $version->id)
            ->where('year', (int)$year)
            ->first();

        return response()->json([
            'message' => 'National score retrieved successfully.',
            'data' => $score
        ]);
    }
}
