<?php

namespace App\Http\Controllers;

use App\Models\PillarScore;
use Illuminate\Http\Request;

class PillarScoreController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $q = PillarScore::with('pillar:id,gii_version_id,pillar_number,name')
            ->orderByDesc('year')
            ->orderBy('gii_pillar_id');

        if ($year = $request->query('year')) {
            $q->where('year', (int)$year);
        }
        if ($pillarId = $request->query('gii_pillar_id')) {
            $q->where('gii_pillar_id', $pillarId);
        }
        if ($status = $request->query('status')) {
            $q->where('status', $status);
        }

        $result = $q->paginate($request->query('per_page', 15));

        return response()->json([
            'message' => 'Pillar scores retrieved successfully.',
            'data' => $result,
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $data = $request->validate([
            'gii_pillar_id' => ['required', 'exists:gii_pillars,id'],
            'year' => ['required', 'integer', 'min:1900', 'max:2100'],
            'score' => ['nullable', 'numeric'],
            'meta' => ['nullable', 'array'],
            'status' => ['nullable', 'in:DRAFT,APPROVED,PUBLISHED'],
        ]);

        $exists = PillarScore::where('gii_pillar_id', $data['gii_pillar_id'])
            ->where('year', $data['year'])
            ->exists();

        if ($exists) {
            return response()->json([
                'message' => 'A pillar score already exists for this pillar and year.'
            ], 422);
        }

        $pillarScore = PillarScore::create([
            'gii_pillar_id' => $data['gii_pillar_id'],
            'year' => $data['year'],
            'score' => $data['score'] ?? null,
            'meta' => $data['meta'] ?? null,
            'status' => $data['status'] ?? 'DRAFT',
        ]);

        return response()->json([
            'message' => 'Pillar score created successfully.',
            'data' => $pillarScore,
        ], 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(PillarScore $pillarScore)
    {
        $pillarScore->load('pillar');

        return response()->json([
            'message' => 'Pillar score retrieved successfully.',
            'data' => $pillarScore,
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, PillarScore $pillarScore)
    {
        $data = $request->validate([
            'gii_pillar_id' => ['sometimes', 'exists:gii_pillars,id'],
            'year' => ['sometimes', 'integer', 'min:1900', 'max:2100'],
            'score' => ['nullable', 'numeric'],
            'meta' => ['nullable', 'array'],
            'status' => ['sometimes', 'in:DRAFT,APPROVED,PUBLISHED'],
        ]);

        $pillarId = $data['gii_pillar_id'] ?? $pillarScore->gii_pillar_id;
        $year = $data['year'] ?? $pillarScore->year;

        $exists = PillarScore::where('gii_pillar_id', $pillarId)
            ->where('year', $year)
            ->where('id', '!=', $pillarScore->id)
            ->exists();

        if ($exists) {
            return response()->json([
                'message' => 'A pillar score already exists for this pillar and year.'
            ], 422);
        }

        $pillarScore->update($data);

        return response()->json([
            'message' => 'Pillar score updated successfully.',
            'data' => $pillarScore->fresh(),
        ]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(PillarScore $pillarScore)
    {
        $pillarScore->delete();

        return response()->json([
            'message' => 'Pillar score deleted successfully.',
        ]);
    }
}
