<?php

namespace App\Http\Controllers;
use App\Models\ApprovalAction;
use App\Models\IndicatorObservation;
use App\Models\Submission;
use App\Models\SubmissionFile;
use App\Models\ValidationResult;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;

class SubmissionController extends Controller
{
    public function index()
    {
        $q = Submission::with(['institution','indicator','files'])
            ->orderByDesc('created_at');

        if ($status = request('status')) $q->where('status', $status);
        if ($year = request('year')) $q->where('year', (int)$year);
        if ($institutionId = request('institution_id')) $q->where('institution_id', $institutionId);

        return response()->json([
            'message' => 'Submissions retrieved successfully.',
            'data' => $q->paginate(request('per_page', 15))
        ]);
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'institution_id' => ['required', 'exists:institutions,id'],
            'gii_indicator_id' => ['nullable', 'exists:gii_indicators,id'],
            'year' => ['required', 'integer', 'min:1900', 'max:2100'],
            'month' => ['nullable', 'integer', 'min:1', 'max:12'],
            'source' => ['nullable', 'string', 'max:100'],
            'notes' => ['nullable', 'string'],
        ]);

        $submission = Submission::create([
            'institution_id' => $data['institution_id'],
            'submitted_by' => auth()->id(),
            'gii_indicator_id' => $data['gii_indicator_id'] ?? null,
            'year' => $data['year'],
            'month' => $data['month'] ?? null,
            'source' => $data['source'] ?? 'portal',
            'status' => 'DRAFT',
            'notes' => $data['notes'] ?? null,
        ]);

        ApprovalAction::create([
            'submission_id' => $submission->id,
            'acted_by' => auth()->id(),
            'action' => 'SUBMIT',
            'comment' => 'Submission created (draft).'
        ]);

        return response()->json([
            'message' => 'Submission created successfully.',
            'data' => $submission
        ], 201);
    }

    public function show(Submission $submission)
    {
        $submission->load(['institution','indicator','files','rawRecords','validations','approvals']);

        return response()->json([
            'message' => 'Submission retrieved successfully.',
            'data' => $submission
        ]);
    }

    public function upload(Request $request, Submission $submission)
    {
        $request->validate([
            'file' => ['required','file','max:20480'], // 20MB
            'file_type' => ['nullable', 'in:evidence,dataset,report,annex'],
        ]);

        $file = $request->file('file');
        $path = $file->store("niims/submissions/{$submission->id}", 'public');
        $checksum = hash_file('sha256', $file->getRealPath());

        $saved = SubmissionFile::create([
            'submission_id' => $submission->id,
            'path' => $path,
            'original_name' => $file->getClientOriginalName(),
            'mime' => $file->getMimeType(),
            'size' => $file->getSize(),
            'file_type' => $request->input('file_type', 'evidence'),
            'checksum' => $checksum,
            'uploaded_by' => $request->user()?->id,
        ]);

        return response()->json([
            'message' => 'File uploaded successfully.',
            'data' => $saved
        ], 201);
    }

    public function addRawRecord(Request $request, Submission $submission)
    {
        $data = $request->validate([
            'payload' => ['required', 'array'],
            'hash' => ['nullable', 'string', 'max:128'],
            'source_reference' => ['nullable', 'string', 'max:255'],
            'status' => ['nullable', 'in:RAW,VALID,INVALID'],
            'notes' => ['nullable', 'string'],
        ]);

        $record = $submission->rawRecords()->create([
            'payload' => $data['payload'],
            'hash' => $data['hash'] ?? null,
            'source_reference' => $data['source_reference'] ?? null,
            'status' => $data['status'] ?? 'RAW',
            'notes' => $data['notes'] ?? null,
        ]);

        return response()->json([
            'message' => 'Raw record added successfully.',
            'data' => $record
        ], 201);
    }

    public function submit(Submission $submission)
    {
        if ($submission->status !== 'DRAFT') {
            return response()->json(['message' => 'Only DRAFT submissions can be submitted.'], 422);
        }

        $submission->update(['status' => 'SUBMITTED']);

        ApprovalAction::create([
            'submission_id' => $submission->id,
            'acted_by' => auth()->id(),
            'action' => 'SUBMIT',
            'comment' => 'Submitted for validation.'
        ]);

        return response()->json([
            'message' => 'Submission submitted for validation.',
            'data' => $submission->fresh()
        ]);
    }

    /**
     * Minimal validation example:
     * - must have at least 1 raw record or 1 file
     * - must have year and institution
     * - if indicator_id present, check if payload has "value" numeric in any row
     */
    public function validateSubmission(Submission $submission)
    {
        if (!in_array($submission->status, ['SUBMITTED','REJECTED'], true)) {
            return response()->json(['message' => 'Submission must be SUBMITTED (or REJECTED) to validate.'], 422);
        }

        DB::transaction(function () use ($submission) {
            // clear old validation results
            $submission->validations()->delete();

            $hasAny = $submission->rawRecords()->exists() || $submission->files()->exists();
            if (!$hasAny) {
                ValidationResult::create([
                    'submission_id' => $submission->id,
                    'level' => 'ERROR',
                    'rule_code' => 'EMPTY_SUBMISSION',
                    'message' => 'Submission must contain at least one raw record or an evidence file.'
                ]);
            }

            if (!$submission->year) {
                ValidationResult::create([
                    'submission_id' => $submission->id,
                    'level' => 'ERROR',
                    'rule_code' => 'MISSING_YEAR',
                    'message' => 'Year is required.'
                ]);
            }

            // Example: if indicator specified, attempt to locate numeric "value" in raw payloads
            if ($submission->gii_indicator_id) {
                $valueFound = $submission->rawRecords()
                    ->whereRaw("JSON_EXTRACT(payload, '$.value') IS NOT NULL")
                    ->exists();

                if (!$valueFound) {
                    ValidationResult::create([
                        'submission_id' => $submission->id,
                        'level' => 'WARNING',
                        'rule_code' => 'VALUE_NOT_FOUND',
                        'message' => 'No numeric value found in raw payload field $.value. Mapping may be required.'
                    ]);
                }
            }

            $hasErrors = $submission->validations()->where('level', 'ERROR')->exists();

            $submission->update([
                'status' => $hasErrors ? 'REJECTED' : 'VALIDATED',
                'rejection_reason' => $hasErrors ? 'Validation failed. Fix errors and resubmit.' : null,
            ]);

            ApprovalAction::create([
                'submission_id' => $submission->id,
                'acted_by' => auth()->id(),
                'action' => $hasErrors ? 'REJECT' : 'VALIDATE',
                'comment' => $hasErrors ? 'Validation failed.' : 'Validation passed.'
            ]);
        });

        return response()->json([
            'message' => 'Validation completed.',
            'data' => $submission->fresh()->load('validations')
        ]);
    }

    public function approve(Submission $submission)
    {
        if ($submission->status !== 'VALIDATED') {
            return response()->json(['message' => 'Only VALIDATED submissions can be approved.'], 422);
        }

        $submission->update(['status' => 'APPROVED']);

        ApprovalAction::create([
            'submission_id' => $submission->id,
            'acted_by' => auth()->id(),
            'action' => 'APPROVE',
            'comment' => 'Submission approved.'
        ]);

        // Minimal observation creation (if indicator is direct-supplied):
        // expects first raw record payload.value
        if ($submission->gii_indicator_id) {
            $first = $submission->rawRecords()->first();
            $value = $first['payload']['value'] ?? null;

            if (is_numeric($value)) {
                IndicatorObservation::updateOrCreate(
                    [
                        'gii_indicator_id' => $submission->gii_indicator_id,
                        'year' => $submission->year,
                        'month' => $submission->month,
                    ],
                    [
                        'submission_id' => $submission->id,
                        'value' => (float)$value,
                        'unit' => $submission->indicator?->unit,
                        'status' => 'APPROVED',
                        'confidence' => 90,
                        'meta' => ['source' => $submission->source],
                    ]
                );
            }
        }

        return response()->json([
            'message' => 'Submission approved successfully.',
            'data' => $submission->fresh()
        ]);
    }

    public function reject(Request $request, Submission $submission)
    {
        $request->validate(['reason' => ['required','string','max:2000']]);

        $submission->update([
            'status' => 'REJECTED',
            'rejection_reason' => $request->reason
        ]);

        ApprovalAction::create([
            'submission_id' => $submission->id,
            'acted_by' => auth()->id(),
            'action' => 'REJECT',
            'comment' => $request->reason
        ]);

        return response()->json([
            'message' => 'Submission rejected.',
            'data' => $submission->fresh()
        ]);
    }
}
