<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::create('national_scores', function (Blueprint $table) {
            $table->id();

            // Which GII framework/version this score belongs to (e.g., 2025 methodology)
            $table->foreignId('gii_version_id')
                ->constrained('gii_versions')
                ->cascadeOnDelete();

            // Reporting year (usually same as GII year, but kept explicit)
            $table->unsignedSmallInteger('year')
                ->comment('Reporting year');

            // Overall national innovation score (computed)
            $table->decimal('score', 10, 4)
                ->nullable()
                ->comment('Computed national score');

            // Optional: ranks and comparative metadata
            $table->unsignedSmallInteger('global_rank')
                ->nullable()
                ->comment('Optional global rank for the year');

            $table->unsignedSmallInteger('regional_rank')
                ->nullable()
                ->comment('Optional regional rank (e.g., Sub-Saharan Africa)');

            // Methodology/breakdown and computation details
            $table->json('meta')
                ->nullable()
                ->comment('Aggregation method, pillar scores snapshot, weights, missing-data flags');

            // Workflow state for public/policy visibility
            $table->enum('status', ['DRAFT', 'APPROVED', 'PUBLISHED'])
                ->default('DRAFT')
                ->comment('Score lifecycle status');

            $table->timestamps();

            // One national score per GII version per year
            $table->unique(
                ['gii_version_id', 'year'],
                'uniq_national_score_version_year'
            );

            // Indexes for dashboards and reporting
            $table->index('year', 'idx_national_score_year');
            $table->index('status', 'idx_national_score_status');
            $table->index('gii_version_id', 'idx_national_score_version');
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('national_scores');
    }
};
